/***************************************************************************
 *
 * Copyright 2012 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include "IpcModule.h"
#include "socketShared.h"
#include <stdio.h>
#include <stdlib.h>  /* getenv */
#include <string.h>  /* memset */
#include <signal.h>
#include <unistd.h>
#include <systemd/sd-daemon.h>

static int connect_systemd_socket()
{
    int fd;
    const char* runtime_dir;
    const char* name = "lmanager-0";
    char path[256] = {0};

    runtime_dir = getenv("LM_SHARED_DIR");
    if (!runtime_dir) {
        return -1;
    }

    snprintf(path, sizeof path, "%s/%s", runtime_dir, name);

    fd = socket(PF_LOCAL, SOCK_STREAM, 0);
    memset(&gState.unSocket.serverAddrUn, 0, sizeof(gState.unSocket.serverAddrUn));
    gState.unSocket.serverAddrUn.sun_family = AF_LOCAL;
    strncpy(gState.unSocket.serverAddrUn.sun_path, path, sizeof(gState.unSocket.serverAddrUn.sun_path)-1);

    if (0 != connect(fd,
                    (struct sockaddr *) &gState.unSocket.serverAddrUn,
                    sizeof(gState.unSocket.serverAddrUn)))
    {
        printf("TcpIpcModule: connection to %s failed.\n", path);
        return -1;
    }

    gState.unSocket.fd = fd;
    gState.unSocket.isInitialized = ILM_TRUE;
    FD_SET(fd, &gState.monitoredSockets);
    gState.monitoredSocketMax = (gState.monitoredSocketMax > fd) ? gState.monitoredSocketMax : fd;

    return 0;
}

static int add_systemd_sockets()
{
    int fd;
    int cnt_systemd_sockets;
    int current_fd = 0;
    const char* runtime_dir;
    const char* name = "lmanager-0";
    char path[256] = {0};

    runtime_dir = getenv("LM_SHARED_DIR");
    if (!runtime_dir) {
        return -1;
    }

    snprintf(path, sizeof path, "%s/%s", runtime_dir, name);

    cnt_systemd_sockets = sd_listen_fds(1);

    if (cnt_systemd_sockets < 0) {
        printf("TcpIpModule: sd_listen_fds failed with: %d" , cnt_systemd_sockets);
        return -1;
    }

    /* socket-based activation not used, return silently */
    if (cnt_systemd_sockets == 0)
        return 0;

    while (current_fd < cnt_systemd_sockets) {
        fd = SD_LISTEN_FDS_START + current_fd;

        if (sd_is_socket_unix(fd, AF_UNIX, 1, path, 0) <= 0) {
            printf("TcpIpModule: invalid socket provided from systemd");
            current_fd++;
            continue;
        }

        gState.unSocket.fd = fd;
        gState.unSocket.isInitialized = ILM_TRUE;
        FD_SET(fd, &gState.monitoredSockets);
        gState.monitoredSocketMax = (gState.monitoredSocketMax > fd) ? gState.monitoredSocketMax : fd;

        current_fd++;
    }

    return current_fd;
}

t_ilm_bool initServiceMode()
{
    t_ilm_bool isClient = ILM_FALSE;
    t_ilm_bool result = ILM_TRUE;
    const char* portString = getenv(ENV_TCP_PORT);
    int port = portString ? atoi(portString) : SOCKET_TCP_PORT;
    struct sigaction sigpipe_action;
    int on = 1;
    struct hostent* incoming_conn;
    const char* hostname = getenv(ENV_TCP_HOST);

    /* ignore broken pipe, if clients disconnect, handled in receive()*/
    sigpipe_action.sa_handler = SIG_IGN;
    sigemptyset(&sigpipe_action.sa_mask);
    sigpipe_action.sa_flags = 0;

    sigaction(SIGPIPE,&sigpipe_action,&gState.sigpipe_old);

    if (add_systemd_sockets() < 0)
    {
        printf("TcpIpcModule: error: could not add systemd sockets");
    }

    gState.isClient = isClient;

    gState.inSocket.fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);

    if (gState.inSocket.fd < 0)
    {
        printf("TcpIpcModule: socket()...failed\n");
        result = ILM_FALSE;
    }

    gState.inSocket.serverAddrIn.sin_family = AF_INET;
    gState.inSocket.serverAddrIn.sin_port = htons(port);
    memset(&(gState.inSocket.serverAddrIn.sin_zero), '\0', 8);

    setsockopt(gState.inSocket.fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));

    if (!hostname)
    {
        hostname = SOCKET_TCP_HOST;
    }

    incoming_conn = gethostbyname(hostname);
    if (!incoming_conn)
    {
        printf("TcpIpcModule: error: could not resolve host '%s'.\n", hostname);
        result = ILM_FALSE;
    }
    else
    {
        memcpy(&gState.inSocket.serverAddrIn.sin_addr.s_addr, incoming_conn->h_addr_list[0], incoming_conn->h_length);
    }

    if (0 > bind(gState.inSocket.fd,
                (struct sockaddr *) &gState.inSocket.serverAddrIn,
                sizeof(gState.inSocket.serverAddrIn)))
    {
        printf("TcpIpcModule: bind()...failed\n");
        result = ILM_FALSE;
    }

    if (listen(gState.inSocket.fd, SOCKET_MAX_PENDING_CONNECTIONS) < 0)
    {
        printf("TcpIpcModule: listen()...failed\n");
        result = ILM_FALSE;
    }

    FD_SET(gState.inSocket.fd, &gState.monitoredSockets);
    gState.monitoredSocketMax = gState.inSocket.fd;
    gState.inSocket.isInitialized = ILM_TRUE;

    return result;
}

t_ilm_bool initClientMode()
{
    t_ilm_bool isClient = ILM_TRUE;
    t_ilm_bool result = ILM_TRUE;
    const char* portString = getenv(ENV_TCP_PORT);
    int port = portString ? atoi(portString) : SOCKET_TCP_PORT;
    struct sigaction sigpipe_action;
    struct hostent* server;
    const char* hostname = getenv(ENV_TCP_HOST);

    /* ignore broken pipe, if clients disconnect, handled in receive()*/
    sigpipe_action.sa_handler = SIG_IGN;
    sigemptyset(&sigpipe_action.sa_mask);
    sigpipe_action.sa_flags = 0;

    sigaction(SIGPIPE,&sigpipe_action,&gState.sigpipe_old);

    gState.isClient = isClient;

    if (!connect_systemd_socket())
    {
        return result;
    }

    gState.inSocket.fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);

    if (gState.inSocket.fd < 0)
    {
        printf("TcpIpcModule: socket()...failed\n");
        result = ILM_FALSE;
    }

    gState.inSocket.serverAddrIn.sin_family = AF_INET;
    gState.inSocket.serverAddrIn.sin_port = htons(port);
    memset(&(gState.inSocket.serverAddrIn.sin_zero), '\0', 8);

    if (!hostname)
    {
        hostname = SOCKET_TCP_HOST;
    }

    server = gethostbyname(hostname);
    if (!server)
    {
        printf("TcpIpcModule: error: could not resolve host '%s'.\n", hostname);
        result = ILM_FALSE;
    }
    else
    {
        memcpy(&gState.inSocket.serverAddrIn.sin_addr.s_addr, server->h_addr_list[0], server->h_length);
    }

    if (0 != connect(gState.inSocket.fd,
                    (struct sockaddr *) &gState.inSocket.serverAddrIn,
                    sizeof(gState.inSocket.serverAddrIn)))
    {
        printf("TcpIpcModule: connection to %s:%d failed.\n",
                hostname, port);
        result = ILM_FALSE;
    }


    FD_SET(gState.inSocket.fd, &gState.monitoredSockets);
    gState.monitoredSocketMax = gState.inSocket.fd;
    gState.inSocket.isInitialized = ILM_TRUE;

    return result;
}

t_ilm_bool destroy()
{
    int socketNumber;
    for (socketNumber = 0; socketNumber <= gState.monitoredSocketMax; ++socketNumber)
    {
        if (FD_ISSET(socketNumber, &gState.monitoredSockets))
        {
            //printf("TcpIpcModule: Closing socket %d\n", socketNumber);
            FD_CLR(socketNumber, &gState.monitoredSockets);
            close(socketNumber);
        }
    }

    /* return to default signal handling */
    sigaction(SIGPIPE, &gState.sigpipe_old,NULL);

    return ILM_TRUE;
}

